from krita import *
from PyQt5.QtWidgets import (
    QDialog,
    QVBoxLayout,
    QLabel,
    QSlider,
    QPushButton,
    QMessageBox,
)
from PyQt5.QtCore import Qt, QByteArray


class PixelizePlugin(Extension):
    def __init__(self, parent):
        super().__init__(parent)

    def setup(self):
        pass

    def createActions(self, window):
        action = window.createAction("pixelize_action", "Pixelize Selection", "tools")
        action.triggered.connect(self.pixelize)

    def pixelize(self):
        kr = Krita.instance()
        document = kr.activeDocument()

        if not document:
            QMessageBox.warning(None, "Pixelize Plugin", "No active document found!")
            return

        selection = document.selection()
        if not selection:
            QMessageBox.warning(None, "Pixelize Plugin", "No selection found!")
            return

        preview_dialog = PixelizeDialog(document, selection)
        preview_dialog.exec_()


class PixelizeDialog(QDialog):
    def __init__(self, document, selection):
        super().__init__()
        self.document = document
        self.selection = selection

        self.setWindowTitle("Pixelize Preview")
        self.layout = QVBoxLayout()

        self.preview_label = QLabel(self)
        self.layout.addWidget(self.preview_label)

        self.slider = QSlider(Qt.Horizontal, self)
        self.slider.setMinimum(1)
        self.slider.setMaximum(50)
        self.slider.setValue(10)
        self.slider.valueChanged.connect(self.update_preview)
        self.layout.addWidget(self.slider)

        self.apply_button = QPushButton("Apply", self)
        self.apply_button.clicked.connect(self.apply_pixelation)
        self.layout.addWidget(self.apply_button)

        self.setLayout(self.layout)
        self.update_preview()

    def update_preview(self):
        pixel_size = self.slider.value()
        preview = self.pixelate(pixel_size, preview=True)
        self.preview_label.setPixmap(QPixmap.fromImage(preview))

    def apply_pixelation(self):
        pixel_size = self.slider.value()
        self.pixelate(pixel_size, preview=False)
        self.accept()

    def pixelate(self, pixel_size, preview):
        x, y, w, h = (
            self.selection.x(),
            self.selection.y(),
            self.selection.width(),
            self.selection.height(),
        )
        layer = self.document.activeNode()

        image_data = layer.pixelData(x, y, w, h)
        image = QImage(image_data, w, h, QImage.Format_ARGB32)

        for px in range(0, w, pixel_size):
            for py in range(0, h, pixel_size):
                color = image.pixelColor(px, py)
                for i in range(px, min(px + pixel_size, w)):
                    for j in range(py, min(py + pixel_size, h)):
                        image.setPixelColor(i, j, color)

        if preview:
            return image.scaled(w, h, Qt.KeepAspectRatio)

        # Convert QImage to bytes for setPixelData
        image_data = QByteArray()
        buffer = image.bits()
        buffer.setsize(image.byteCount())
        image_data.append(buffer.asstring())

        layer.setPixelData(image_data, x, y, w, h)
        self.document.refreshProjection()


Krita.instance().addExtension(PixelizePlugin(Krita.instance()))
