from krita import *
from PyQt5.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit, QPushButton, QComboBox

import random, math

def random_point(max_x, max_y):
    return (random.randint(0, max_x), random.randint(0, max_y))

def random_color_bgra():
    return [random.randint(0, 255), random.randint(0, 255), random.randint(0, 255), 255]

def euclid_distance(a, b):
    return math.sqrt((a[0] - b[0])**2 + (a[1] - b[1])**2)

def find_closest_point(points, p):
    distances = [euclid_distance(point, p) for point in points]
    min_i = distances.index(min(distances))

    distances.sort()
    return min_i, distances[0] / distances[1]

def shade_color(color, t):
    return (int(color[0] * t), int(color[1] * t), int(color[2] * t), int(color[3]))

def smoothstep(t):
    return t * t * (3 - 2*t)

def inverse_smoothstep(t):
    return 0.5 - math.sin(math.asin(1.0 - 2.0 * t) / 3.0)


class VoronoiGenerationDialog(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Voronoi Texture Options")
        self.setGeometry(100, 100, 400, 200)
        
        # Layouts
        layout = QVBoxLayout()
        
        # Resolution inputs
        resolution_layout = QHBoxLayout()
        resolution_layout.addWidget(QLabel("Resolution (Width x Height):"))
        self.width_input = QLineEdit("512")
        self.height_input = QLineEdit("512")
        resolution_layout.addWidget(self.width_input)
        resolution_layout.addWidget(self.height_input)
        layout.addLayout(resolution_layout)
        
        # Number of points input
        points_layout = QHBoxLayout()
        points_layout.addWidget(QLabel("Number of Points:"))
        self.points_input = QLineEdit("10")
        points_layout.addWidget(self.points_input)
        layout.addLayout(points_layout)
        
        # Function selection
        function_layout = QHBoxLayout()
        function_layout.addWidget(QLabel("Function:"))
        self.function_select = QComboBox()
        self.function_select.addItems(["smoothstep", "normal", "inverse smoothstep"])
        function_layout.addWidget(self.function_select)
        layout.addLayout(function_layout)
        
        # Buttons
        button_layout = QHBoxLayout()
        self.ok_button = QPushButton("OK")
        self.cancel_button = QPushButton("Cancel")
        button_layout.addWidget(self.ok_button)
        button_layout.addWidget(self.cancel_button)
        layout.addLayout(button_layout)
        
        self.setLayout(layout)
        
        # Button connections
        self.ok_button.clicked.connect(self.accept)
        self.cancel_button.clicked.connect(self.reject)
    
    def get_values(self):
        return {
            "width": int(self.width_input.text()),
            "height": int(self.height_input.text()),
            "points": int(self.points_input.text()),
            "function": self.function_select.currentText(),
        }

class VoronoiTexture(Extension):

    n_points = 10
    width = 512
    height = 512
    function = 'normal'

    def __init__(self, parent):
        # This is initialising the parent, always important when subclassing.
        super().__init__(parent)

    def setup(self):
        pass

    def createActions(self, window):
        action = window.createAction("generate_voronoi_texture", "Generate Voronoi Texture", "tools")
        action.triggered.connect(self.generateVoronoiTexture)
    
    def generateVoronoiTexture(self):
        if (not self.prompt_user()):
            return

        kr = Krita.instance()
        doc = kr.createDocument(self.width, self.height, "Voronoi", "RGBA", "U8", "", 120.0)

        # Open the image
        kr.activeWindow().addView(doc)

        # Get the pixel data from the first layer (new document -> guaranteed to have exactly one layer)
        layer = doc.rootNode().childNodes()[0]
        pixel_data = layer.pixelData(0, 0, doc.width(), doc.height())
        
        width, height = doc.width(), doc.height()
        pixels = bytearray(pixel_data)  # Convert to mutable byte array

        random_points = [random_point(self.width, self.height) for x in range(self.n_points)]
        random_colors = [random_color_bgra() for x in range(self.n_points)]

        # Shade cells
        for y in range(height):
            for x in range(width):
                index = (y * width + x) * 4  # RGBA pixels in a flat array

                closest_i, t = find_closest_point(random_points, (x, y))
                if self.function == 'inverse smoothstep':
                    pixels[index:index+4] = shade_color(random_colors[closest_i], inverse_smoothstep(1 - t))
                elif self.function == 'smoothstep':
                    pixels[index:index+4] = shade_color(random_colors[closest_i], smoothstep(1 - t))
                else:
                    pixels[index:index+4] = shade_color(random_colors[closest_i], 1 - t)

        # Write modified pixels back to the layer - construct raw bytes from pixels mutable byte array
        layer.setPixelData(bytes(pixels), 0, 0, width, height)
        
        # Refresh the canvas to show the updated layer
        doc.refreshProjection()
    
    def prompt_user(self):
        dialog = VoronoiGenerationDialog()
        if dialog.exec_() == QDialog.Accepted:
            # Get user inputs
            options = dialog.get_values()
            self.width, self.height, self.n_points, self.function = options["width"], options["height"], options["points"], options["function"]
            return True
        return False


# And add the extension to Krita's list of extensions:
Krita.instance().addExtension(VoronoiTexture(Krita.instance()))